<?php
// LicenseServer/api/reset.php
// Reset (unbind) a license's bound domain. Requires admin_api_token from config.php.
//
// Request (POST JSON):
// { "purchase_code": "ENV-XXXX-XXXXXX", "token": "ADMIN_API_TOKEN" }
//
// Response:
// { "ok": true }  OR  { "ok": false, "error": "invalid_token|missing_code|not_found" }

header('Content-Type: application/json; charset=utf-8');

$input = json_decode(file_get_contents('php://input'), true) ?? [];

// Basic validation
$code  = trim((string)($input['purchase_code'] ?? ''));
$token = trim((string)($input['token'] ?? ''));
if ($code === '') {
  http_response_code(400);
  echo json_encode(['ok' => false, 'error' => 'missing_code']);
  exit;
}

// Load config + DB
$cfgPath = dirname(__DIR__) . '/config.php';
$dbPath  = dirname(__DIR__) . '/license_db.php';
if (!is_file($cfgPath) || !is_file($dbPath)) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'server_misconfigured']);
  exit;
}
$cfg = require $cfgPath;
require $dbPath;

// Verify admin token
if (!isset($cfg['admin_api_token']) || !hash_equals($cfg['admin_api_token'], $token)) {
  http_response_code(403);
  echo json_encode(['ok' => false, 'error' => 'invalid_token']);
  exit;
}

try {
  // Look up the license
  $stmt = $db->prepare("SELECT id FROM licenses WHERE purchase_code = ? LIMIT 1");
  $stmt->execute([$code]);
  $row = $stmt->fetch();

  if (!$row) {
    http_response_code(404);
    echo json_encode(['ok' => false, 'error' => 'not_found']);
    exit;
  }

  $db->beginTransaction();
  // Unbind domain & optionally set status to active
  $stmt = $db->prepare("UPDATE licenses SET bound_domain = NULL, status = 'active' WHERE purchase_code = ? LIMIT 1");
  $stmt->execute([$code]);

  // Optional: log event
  $db->exec("CREATE TABLE IF NOT EXISTS license_events (
      id INT AUTO_INCREMENT PRIMARY KEY,
      license_id INT NOT NULL,
      event_type ENUM('bind','verify','reset') NOT NULL,
      domain VARCHAR(255) DEFAULT NULL,
      user_agent VARCHAR(255) DEFAULT NULL,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      INDEX idx_license (license_id),
      INDEX idx_created (created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
  $stmt = $db->prepare("INSERT INTO license_events (license_id, event_type, domain, user_agent) VALUES (?,?,?,?)");
  $stmt->execute([$row['id'], 'reset', null, $_SERVER['HTTP_USER_AGENT'] ?? null]);

  $db->commit();

  echo json_encode(['ok' => true]);
} catch (Throwable $e) {
  if ($db->inTransaction()) $db->rollBack();
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'server_error']);
}
