<?php
// license_server/api/verify.php
header('Content-Type: application/json; charset=utf-8');

require __DIR__ . '/../license_db.php';
$cfg = require __DIR__ . '/../config.php';

// BASIC CORS
$origin = $_SERVER['HTTP_ORIGIN'] ?? '*';
if ($cfg['allow_origins'] === ['*'] || in_array($origin, $cfg['allow_origins'], true)) {
  header("Access-Control-Allow-Origin: $origin");
  header("Vary: Origin");
}
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  header('Access-Control-Allow-Methods: POST, OPTIONS');
  header('Access-Control-Allow-Headers: Content-Type');
  exit;
}

// INPUT
$input = json_decode(file_get_contents('php://input'), true) ?? [];
$purchase_code = trim((string)($input['purchase_code'] ?? ''));
$domain        = strtolower(trim((string)($input['domain'] ?? '')));

if ($purchase_code === '' || $domain === '') {
  http_response_code(400);
  echo json_encode(['ok'=>false, 'error'=>'bad_request']);
  exit;
}

// LOOKUP LICENSE
$stmt = $pdo->prepare('SELECT * FROM licenses WHERE purchase_code = ? LIMIT 1');
$stmt->execute([$purchase_code]);
$row = $stmt->fetch();

if (!$row) {
  http_response_code(404);
  echo json_encode(['ok'=>false, 'error'=>'invalid_code']);
  exit;
}
if ($row['status'] !== 'active') {
  http_response_code(403);
  echo json_encode(['ok'=>false, 'error'=>'inactive_license']);
  exit;
}

// BIND OR CHECK DOMAIN
$bound = strtolower(trim((string)$row['bound_domain']));
if ($bound === '' || $bound === null) {
  $upd = $pdo->prepare('UPDATE licenses SET bound_domain=?, activations=activations+1, last_seen=NOW() WHERE id=?');
  $upd->execute([$domain, $row['id']]);
  $bound = $domain;
} else {
  $norm = function($h){ return preg_replace('~^www\.~i', '', $h); };
  if ($norm($domain) !== $norm($bound)) {
    http_response_code(409);
    echo json_encode(['ok'=>false, 'error'=>'bound_to_other_domain', 'bound_domain'=>$row['bound_domain']]);
    exit;
  }
  $pdo->prepare('UPDATE licenses SET last_seen=NOW() WHERE id=?')->execute([$row['id']]);
}

// BUILD TOKEN
$now = time();
$exp = $now + (int)$cfg['token_ttl'];
$payload = json_encode([
  'product' => $row['product_slug'],
  'purchase_code' => $purchase_code,
  'domain' => $bound,
  'iat' => $now,
  'exp' => $exp
], JSON_UNESCAPED_SLASHES);

// SIGN
$priv = @openssl_pkey_get_private(@file_get_contents($cfg['private_key']));
if (!$priv) {
  http_response_code(500);
  echo json_encode(['ok'=>false, 'error'=>'signing_key_missing']);
  exit;
}
openssl_sign($payload, $sig, $priv, OPENSSL_ALGO_SHA256);
openssl_free_key($priv);

$enc = function($s){ return rtrim(strtr(base64_encode($s), '+/', '-_'), '='); };
$token = $enc($payload).'.'.$enc($sig);

echo json_encode(['ok'=>true, 'token'=>$token, 'bound_domain'=>$bound]);
